import settings from 'electron-settings';
import axios from 'axios';
import { Logger } from './Logger';
import { OS_IS_MAC, appVersion } from '../constants/versions';
import getLicense from './headset/license';
import { DB_MODEL_NAME } from './headset/consts';

const KEY_USER_DETAILS = 'user_details';
const KEY_USER_EMAIL = 'user_email';
const KEY_USER_TOKEN = 'user_token';
const KEY_FURTHER_REGISTRATION_REQUIRED = 'further_registration_required';

const STAGE_DEV = 'dev';
const STAGE_RC = 'rc';
const STAGE_PROD = 'prod';

const license = getLicense();

let STAGE = STAGE_PROD;

if (license && license.stage) {
    STAGE = license.stage;
}

const STAGE_TO_URL = {
    [STAGE_DEV]:
        'https://whyo3fjk7d.execute-api.us-east-1.amazonaws.com/dev/api/v1/',
    [STAGE_RC]:
        'https://whyo3fjk7d.execute-api.us-east-1.amazonaws.com/rc/api/v1/',
    [STAGE_PROD]:
        'https://whyo3fjk7d.execute-api.us-east-1.amazonaws.com/prod/api/v1/'
};

const BASE_API_URL = STAGE_TO_URL[STAGE];
console.log('Using stage', STAGE, BASE_API_URL);

export const API_ERROR_NETWORK = 'network_error';
export const API_ERROR_USER_ALREADY_EXISTS = 'user_exists';
export const API_ERROR_USER_NOT_FOUND = 'user_not_found';
export const API_ERROR_DB_ERROR = 'db_error';
export const API_ERROR_EMAIL_NOT_SENT = 'email_not_sent';
export const API_ERROR_INVALID_EMAIL_CODE = 'invalid_email_code';
export const API_ERROR_MAX_VERIFICATIONS_REACHED = 'max_verifications_reached';

export const CHANNEL_ON_AUTHENTICATION_CHANGED = 'on_authentication_changed';

/** Returns user details */
export function getUserDetails() {
    return settings.getSync(KEY_USER_DETAILS);
}

export function isSignedIn() {
    return settings.getSync(KEY_USER_EMAIL);
}

export function getUserEmail() {
    return settings.getSync(KEY_USER_EMAIL);
}

export function signOut() {
    const userToken = settings.getSync(KEY_USER_TOKEN);

    // Let the server know the user signed out
    Logger.info(`signOut`);
    apiCall('signout', { user_token: userToken }, false)
        .then(() => true)
        .catch(exc => {
            console.error(exc);
        });

    settings.unsetSync(KEY_USER_DETAILS);
    settings.unsetSync(KEY_USER_EMAIL);
    settings.unsetSync(KEY_USER_TOKEN);
    settings.unsetSync(KEY_FURTHER_REGISTRATION_REQUIRED);
}

export function isWaitingForEmailVerification() {
    return (
        settings.getSync(KEY_USER_EMAIL) &&
        (!settings.getSync(KEY_USER_TOKEN) ||
            settings.getSync(KEY_FURTHER_REGISTRATION_REQUIRED))
    );
}

async function apiCall(endpoint, params, checkError = true) {
    return new Promise((resolve, reject) => {
        Logger.info(`Calling ${endpoint}: ${JSON.stringify(params)}`);

        const customHeaders = {
            'X-CLIENT-ID': getPlatformOs(),
            'X-CLIENT-VERSION': appVersion
        };

        axios
            .post(`${BASE_API_URL}${endpoint}`, params, {
                headers: customHeaders
            })
            .then(response => {
                Logger.info(
                    `API response: ${response.status} - ${JSON.stringify(
                        response.data
                    )}`
                );

                if (checkError && !response.data.success) {
                    reject(response.data.error);
                } else {
                    resolve(response.data);
                }

                return true;
            })
            .catch(exc => {
                Logger.error(exc);
                reject(API_ERROR_NETWORK);
            });
    });
}

export async function checkIfEmailExists(email) {
    return new Promise((resolve, reject) => {
        Logger.info(`checkIfEmailExists ${email}`);
        apiCall('email_exists', { email }, false)
            .then(response => {
                resolve(response.exists);
                return true;
            })
            .catch(error => {
                reject(error);
            });
    });
}

export function getPlatformOs() {
    return OS_IS_MAC ? 'cardo-update-macos' : 'cardo-update-windows';
}

export async function signIn(email, language) {
    settings.setSync(KEY_USER_EMAIL, null);

    return new Promise((resolve, reject) => {
        Logger.info(`signIn ${email}: ${language}`);
        apiCall('signin', { email, language, platform_os: getPlatformOs() })
            .then(_ => {
                settings.setSync(KEY_USER_EMAIL, email.toLowerCase());
                resolve(true);
                return true;
            })
            .catch(error => {
                reject(error);
            });
    });
}

export async function signUp(
    email,
    firstName,
    lastName,
    country,
    language,
    marketingApproval
) {
    settings.setSync(KEY_USER_EMAIL, null);

    return new Promise((resolve, reject) => {
        Logger.info(`signup ${email}: ${language}`);
        apiCall('signup', {
            email,
            first_name: firstName,
            last_name: lastName,
            country,
            language,
            marketing_approval: marketingApproval,
            platform_os: getPlatformOs()
        })
            .then(_ => {
                settings.setSync(KEY_USER_EMAIL, email.toLowerCase());
                resolve(true);
                return true;
            })
            .catch(error => {
                reject(error);
            });
    });
}

function refreshUserDetails() {
    Logger.info('refreshUserDetails');

    apiCall(
        'get_user_info',
        {
            user_token: settings.getSync(KEY_USER_TOKEN)
        },
        false
    )
        .then(response => {
            settings.setSync(KEY_USER_DETAILS, {
                first_name: response.first_name,
                last_name: response.last_name,
                marketing_approval: response.marketing_approval
            });

            return true;
        })
        .catch(error => {
            Logger.error(error);
        });
}

export async function checkIfSignedIn() {
    return new Promise((resolve, reject) => {
        if (!isWaitingForEmailVerification()) reject();
        const email = settings.getSync(KEY_USER_EMAIL);
        if (!email) reject();

        Logger.info(`checkIfSignedIn ${email}`);

        apiCall('get_token', { email, platform_os: getPlatformOs() })
            .then(response => {
                if (!response.user_token) {
                    // User not signed in yet
                    console.log('checkIfSignIn - user not signed in yet');
                    resolve([false]);
                } else {
                    console.log(
                        `checkIfSignIn - user signed in - ${response.user_token} - response: ${response}`
                    );
                    settings.setSync(KEY_USER_TOKEN, response.user_token);

                    if (response.registration_required) {
                        settings.setSync(
                            KEY_FURTHER_REGISTRATION_REQUIRED,
                            true
                        );
                    } else {
                        refreshUserDetails();
                    }
                    resolve([true, response]);
                }
                return true;
            })
            .catch(error => {
                reject(error);
            });
    });
}

export async function setUserInfo(info) {
    return new Promise((resolve, reject) => {
        Logger.info(`setUserInfo ${info}`);

        apiCall('set_user_info', {
            user_token: settings.getSync(KEY_USER_TOKEN),
            ...info
        })
            .then(response => {
                if (!response.success) {
                    console.log(`Error: ${response}`);
                    reject(response.error);
                } else {
                    settings.setSync(KEY_FURTHER_REGISTRATION_REQUIRED, false);
                    refreshUserDetails();
                    resolve(true);
                }
                return true;
            })
            .catch(error => {
                reject(error);
            });
    });
}

export async function deviceConnected(device) {
    return new Promise((resolve, reject) => {
        Logger.info('deviceConnected', device);

        const deviceParams = {
            user_token: settings.getSync(KEY_USER_TOKEN),
            serial: device.serial,
            model_name: DB_MODEL_NAME[device.model] || device.model,
            fw_version: String(device.fwVersion),
            platform_os: getPlatformOs()
        };

        if (device.deviceLanguage) {
            deviceParams.language = device.deviceLanguage;
        }

        apiCall('device_connected', deviceParams)
            .then(response => {
                if (!response.success) {
                    console.log(`Error: ${response}`);
                    reject(response.error);
                } else {
                    resolve(true);
                }
                return true;
            })
            .catch(error => {
                reject(error);
            });
    });
}

export async function deviceFWUpdated(
    device,
    newFWVersion,
    previousFWVersion,
    installationSuccessful,
    errorCode,
    errorMessage
) {
    return new Promise((resolve, reject) => {
        Logger.info(`deviceFWUpdated ${device}`);

        const params = {
            user_token: settings.getSync(KEY_USER_TOKEN),
            serial: device.serial,
            model_name: DB_MODEL_NAME[device.model] || device.model,
            fw_version: String(newFWVersion),
            previous_fw_version: String(previousFWVersion),
            platform_os: getPlatformOs(),
            installation_status: installationSuccessful
        };

        if (!installationSuccessful) {
            params.error_code = errorCode;
            params.error_message = errorMessage;
        }

        apiCall('firmware_update', params)
            .then(response => {
                if (!response.success) {
                    console.log(`Error: ${response}`);
                    reject(response.error);
                } else {
                    resolve(true);
                }
                return true;
            })
            .catch(error => {
                reject(error);
            });
    });
}
